package pkg

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"time"
)

// creates http client for ensuring timely timeouts
var client = http.Client{
	Timeout:       15 * time.Second,
}

// allowing access to "overloaded" functions
func unfold(m mashup, response *http.Response) error {
	return m.unmarshal(response)
}

// basically overloading the function
type mashup interface {
	unmarshal(response *http.Response) error
}

func (c *tempCountry) unmarshal(resp *http.Response) error {
	err := json.NewDecoder(resp.Body).Decode(&c)
	defer resp.Body.Close()
	return err
}

func (r *response) unmarshal(resp *http.Response) error {
	err := json.NewDecoder(resp.Body).Decode(&r)
	defer resp.Body.Close()
	return err
}

func (s *Specie) unmarshal(resp *http.Response) error {
	err := json.NewDecoder(resp.Body).Decode(&s)
	defer resp.Body.Close()
	return err
}

// parses http.response as a string used later for parsing
func getBody(url string, m mashup) error {

	// sends a http request
	resp, err := client.Get(url)
	if err != nil {
		return err
	}

	// decodes the http response & closes the body
	err = unfold(m, resp)
	if err != nil {
		return err
	}

	// sends appropriate error
	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("%d", resp.StatusCode)
	}

	// all went well
	return nil
}

// func for returning correct errors & response code
func HttpError(w http.ResponseWriter, err error) {

	// if timeout happened
	if strings.Contains(err.Error(), "Client.Timeout exceeded") {
		http.Error(w, "Request timeout", http.StatusGatewayTimeout)
		return // return if error is already displayed
	}

	switch err.Error() {
	case "400":
		http.Error(w, "Bad Request: " + err.Error(), http.StatusBadRequest)
	case "404":
		http.Error(w, "Not found: " + err.Error(), http.StatusNotFound)
	default:
		http.Error(w, "Error occurred: " + err.Error(), http.StatusInternalServerError)
	}

}
