package main

import (
	"bufio"
	"context"
	"fmt"
	"go.mongodb.org/mongo-driver/mongo/options"
	projectdb "llama/database"
	"llama/modules/compiler_v2/go_compiler_v2"
	"llama/modules/extraction"
	ollamaimplementation "llama/modules/ollama-implementation"
	promptlist "llama/modules/prompt_list"
	"llama/program"
	"llama/webServer"
	"log"
	"os"
	"strings"
)

// const Model string = "codellama"
const Model string = "llama3.1"

type NoUI struct{}

// WriteJSON A mock implementation when there are no UI elements
func (n *NoUI) WriteJSON(_ any) error {
	return nil
}

func main() {

	clientOptions := options.Client().ApplyURI(projectdb.UriDb)
	db := projectdb.GetDB(clientOptions, projectdb.Production)
	defer db.CloseDB()

	runProgramWithUI(db)
	// runWithConsoleInput(db)
	// runWithPromptFile(db)
}

func runProgramWithUI(db projectdb.IDataBase) {
	defer db.CloseDB()

	languageModel := &ollamaimplementation.Ollama{Model: Model}

	prog := program.Program{
		LanguageModel: languageModel,
		Ctx:           nil, // Is set in the UIProgram.Run() method
		UI:            nil, // Is set in the UIProgram.Run() method
		DB:            db,
		Compiler:      go_compiler_v2.NewGoCompiler().CheckCompileErrors,
		Extractor:     &extraction.GoExtractor{},
	}

	uiProgram := webServer.UIProgram{Program: prog}
	uiProgram.Run()
}

func runWithConsoleInput(db projectdb.IDataBase) {
	reader := bufio.NewReader(os.Stdin)

	fmt.Print("Enter your prompt (or type 'exit' to quit): ")
	userPrompt, _ := reader.ReadString('\n')
	userPrompt = strings.TrimSpace(userPrompt)

	runWithPrompt(userPrompt, db)
}

func runWithPromptFile(db projectdb.IDataBase) {

	promptList := promptlist.PromptList("../data/promptList.txt")
	if promptList == nil {
		log.Fatal("Prompt list is empty")
	}

	for _, v := range promptList {
		fmt.Println(v)
		runWithPrompt(v, db)
	}
}

func runWithPrompt(userPrompt string, db projectdb.IDataBase) {

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	languageModel := &ollamaimplementation.Ollama{Model: Model}

	prog := &program.Program{
		LanguageModel: languageModel,
		Ctx:           ctx,
		UI:            &NoUI{},
		DB:            db,
		Compiler:      go_compiler_v2.NewGoCompiler().CheckCompileErrors,
		Extractor:     &extraction.GoExtractor{},
	}

	prog.Run(userPrompt, extraction.GoPrompt)
	//go RunProgram(userPrompt, conversationContext, extraction.RustPrompt, rust_compiler_v2.NewRustCompiler().CheckCompileErrors)
}
