package extraction

import (
	"fmt"
	"regexp"
	"strings"
)

var GoPrompt = "The code should be in the Go programming language. There should also be 3 robust test cases within the same code. There should also be a main function inside of which all the execution takes place. Please only provide the source code and no further explanation, The format should be ```go <yourcode + testcases> ```"

var RustPrompt = "The code should be in the Rust programming language. There should also be 3 robust test cases within the same code. There should also be a main function inside of which all the execution takes place. Please only provide the source code and no further explanation, The format should be ```rust <yourcode + testcases> ```"

// func ExtractCode(output string) string {
// 	parts := strings.Split(output, "```")
// 	var extracted = ""
// 	if strings.Contains(parts[1], "rust") {
// 		extracted = strings.TrimLeft(parts[1], "rust")
// 	} else {
// 		extracted = strings.TrimLeft(parts[1], "go")
// 	}
// 	return extracted
// }

type IExtractor interface {
	ExtractCode(text string) (string, error)
	ExtractTestCases(string) string
}

// ExtractCode extracts the code snippet between ``` and removes the language identifier.
func ExtractCode(text string) (string, error) {
	parts := strings.Split(text, "```")
	if len(parts) < 2 {
		return "", fmt.Errorf("the string wasn't in a proper format") // Handle the case if format is incorrect: Return empty string
	}

	// Trim the language identifier like `go` or `rust` from the code
	code := parts[1]
	lines := strings.SplitN(code, "\n", 2)
	if len(lines) > 1 {
		return "\n" + lines[1], nil // Return the code without the first line (language identifier)
	}
	return "", fmt.Errorf("the string doesn't contain any lines")
}

type RustExtractor struct{}

func (re *RustExtractor) ExtractCode(text string) (string, error) {
	return ExtractCode(text)
}

// ExtractTestCases extracts the test cases from the code snippet using regex.
func (re *RustExtractor) ExtractTestCases(text string) string {
	regex := regexp.MustCompile(`^\s*#\[\s*test\s*]\s*\n?\s*fn\s+[a-zA-Z0-9_]+\s*\([^)]*\)\s*\{[^{}]*}`)
	testCases := regex.FindAllString(text, -1)
	result := strings.Join(testCases, "\n")
	return result
}

type GoExtractor struct{}

func (ge *GoExtractor) ExtractCode(text string) (string, error) {
	return ExtractCode(text)
}

// ExtractTestCases extracts the test cases from the code snippet using regex.
func (ge *GoExtractor) ExtractTestCases(text string) string {
	re := regexp.MustCompile(`(?m)^func\s+(Test\w+)\s*\(t\s+\*testing\.T\)\s*{[\s\S]*?^}`)
	testCases := re.FindAllString(text, -1)
	result := strings.Join(testCases, "\n")
	return result
}
