package program

import (
	"context"
	"fmt"
	projectdb "llama/database"
	displayindicator "llama/modules/display-indicator"
	"llama/modules/extraction"
	"log"
	"time"
)

const MaxIterations = 5

type compilerFunc func([]byte, ...string) ([]byte, error)

type ResponseData struct {
	Iteration            uint   `json:"iteration"`
	GeneratedCode        string `json:"generated_code"`
	CompilerOutput       string `json:"compiler_output"`
	CompiledSuccessfully bool   `json:"compiled_successfully"`
	TotalExecutionTime   string `json:"total_execution_time,omitempty"`
	TestCases            string `json:"test_cases,omitempty"`
}

type ILanguageModel interface {
	GetResponse(prompt string, context []int) (string, []int, error) // Response, context, error
}

type IUserInterface interface {
	WriteJSON(v any) error
}

type Program struct {
	LanguageModel ILanguageModel
	Ctx           context.Context
	UI            IUserInterface
	DB            projectdb.IDataBase
	Compiler      compilerFunc
	Extractor     extraction.IExtractor
}

func (p *Program) Run(userPrompt string, languagePrompt string) {
	var currentConversationContext []int
	var currentIteration uint = 1
	startTime := time.Now() // Track start time for execution duration

	currentPrompt := userPrompt
	for {
		select {
		case <-p.Ctx.Done():
			fmt.Println("Process canceled by client")
			return // Exit the loop if context is canceled
		default:

			newPrompt, newContext, exit := p.RunIteration(userPrompt, currentPrompt, languagePrompt, currentConversationContext, currentIteration, startTime)
			if exit {
				return
			}
			currentPrompt = newPrompt
			currentConversationContext = newContext
			currentIteration++
		}
	}
}

// TODO: data writing should not be in this function, this function should just run a single iteration and return all the results
func (p *Program) RunIteration(
	initialPrompt string,
	currentPrompt string,
	languagePrompt string,
	currentConversationContext []int,
	currentIteration uint,
	startTime time.Time,
) (updatedPrompt string, updatedContext []int, shouldExit bool) {
	response, updatedContext, err := generateLLMResponse(p.LanguageModel, currentPrompt, languagePrompt, currentConversationContext)
	if err != nil {
		writeMessage("Error generating response", p.UI)
		return "", updatedContext, true
	}

	// ExtractCode code from the response
	generatedCode, errExtract := extraction.ExtractCode(response)
	if errExtract != nil {
		writeMessage("Improper LLM response", p.UI)
		return "", updatedContext, true
	}

	// Compile the generated code
	output, err := p.Compiler([]byte(generatedCode))
	compilationSuccess := err == nil // Compilation is successful if there is no error

	// Write the response data to the client
	if err := writeResponseData(currentIteration, generatedCode, output, compilationSuccess, startTime, p.UI); err != nil {
		log.Println("Error writing JSON to connection, exiting with error: ", err)
		return "", updatedContext, true
	}

	// If the compilation fails, or the max iterations are reached, return the errors
	if !compilationSuccess && currentIteration < MaxIterations {
		// Update prompt with errors for next iteration
		return "Following are the errors, please fix the code:\n" + string(output), updatedContext, false
	}

	// Checking for max iterations
	if currentIteration >= MaxIterations {
		writeMessage("Max iterations reached", p.UI)
	}

	// Write the data to the database
	p.writeDataToDatabase(initialPrompt, generatedCode, output, startTime, currentIteration, p.DB)

	return "", updatedContext, true
}

func writeMessage(message string, ui IUserInterface) {
	log.Println(message)
	err := ui.WriteJSON(ResponseData{CompilerOutput: message})
	if err != nil {
		log.Println("Error writing JSON to connection, exiting with error: ", err)
	}
}

func writeResponseData(currentIteration uint, generatedCode string, output []byte, compilationSuccess bool, startTime time.Time, ui IUserInterface) error {
	// Prepare response data
	responseData := ResponseData{
		Iteration:            currentIteration,
		GeneratedCode:        generatedCode,
		CompilerOutput:       string(output),
		CompiledSuccessfully: compilationSuccess,
		TotalExecutionTime:   time.Since(startTime).String(),
	}

	// Send iteration data back to the client
	err2 := ui.WriteJSON(responseData)
	return err2
}

func (p *Program) writeDataToDatabase(initialPrompt string, generatedCode string, output []byte, startTime time.Time, currentIteration uint, db projectdb.IDataBase) {
	// ExtractCode test cases
	testCases := p.Extractor.ExtractTestCases(generatedCode)

	resultData := projectdb.ProgramResultData{
		InitialInputQuery: initialPrompt,
		SolutionCode:      generatedCode,
		CompilerOutput:    string(output),
		TestCases:         testCases,
	}

	db.WriteDataToDatabase(resultData)

	// Research data
	researchData := projectdb.ResearchData{
		TimeTakenSeconds: time.Since(startTime).Seconds(),
		Iterations:       int(currentIteration),
	}
	db.WriteResearchDataToDatabase(researchData)
}

func generateLLMResponse(languageModel ILanguageModel, userPrompt string, languagePrompt string, currentConversationContext []int) (string, []int, error) {
	// Prepare the prompt for LLM
	fmt.Println("Prompt received. Generating response...")
	// Display loading indicator (for testing purposes)
	done := make(chan bool)
	go displayindicator.DisplayLoadingIndicator(done)

	// Creating a modified prompt by combining user prompt and language prompt
	modifiedPrompt := userPrompt + languagePrompt
	// Generate LLM response
	response, updatedContext, err := languageModel.GetResponse(modifiedPrompt, currentConversationContext)
	done <- true
	return response, updatedContext, err
}
