let socket;
let isProcessing = false;
let prompts = []; // Store prompts from uploaded file in list mode
let currentPromptIndex = 0; // Track current prompt index in list mode
let currentMode = "single"; // Track whether we are in "single" or "list" mode

// Switch to single prompt mode
function showPromptInput() {
  resetProcess(); // Reset any ongoing process
  document.getElementById("singlePromptSection").style.display = "block";
  document.getElementById("fileUploadSection").style.display = "none";
  currentMode = "single";
}

// Switch to prompt list upload mode
function showFileUpload() {
  resetProcess(); // Reset any ongoing process
  document.getElementById("singlePromptSection").style.display = "none";
  document.getElementById("fileUploadSection").style.display = "block";
  currentMode = "list";
}

// Start processing a single prompt
function startSinglePromptProcess() {
  const userPrompt = document.getElementById("userPrompt").value;
  if (!userPrompt || isProcessing) return;

  isProcessing = true;
  startWebSocket(userPrompt);
}

// Handle file upload and start processing the list of prompts
function uploadAndProcessFile() {
  const fileInput = document.getElementById("promptFile").files[0];
  if (!fileInput) return alert("Please upload a .txt file");

  const reader = new FileReader();
  reader.onload = function (event) {
    // Split file content by line, filter empty lines, and reset processing index
    prompts = event.target.result
      .split(/\r?\n/)
      .filter((line) => line.trim() !== "");
    currentPromptIndex = 0;
    if (prompts.length > 0) {
      processNextPrompt();
    } else {
      alert("File is empty");
    }
  };
  reader.readAsText(fileInput);
}

// Start WebSocket and send initial prompt
function startWebSocket(prompt) {
  // Close any existing WebSocket connection
  if (socket) socket.close();

  // Show loading indicator based on mode
  document.getElementById(
    currentMode === "single" ? "loadingIndicator" : "fileLoadingIndicator"
  ).style.display = "inline-block";

  // Establish a new WebSocket connection
  socket = new WebSocket("ws://localhost:8080/ws");

  socket.onopen = () => {
    socket.send(JSON.stringify({ prompt: prompt }));
  };

  socket.onmessage = (event) => {
    const data = JSON.parse(event.data);
    displayIteration(data);

    // If compilation succeeds and in list mode, move to the next prompt
    if (data.compiled_successfully && currentMode === "list") {
      currentPromptIndex++;
      socket.close(); // Close WebSocket to reset for next prompt
      processNextPrompt();
    } else if (data.compiled_successfully) {
      isProcessing = false; // End single prompt mode
      document.getElementById("loadingIndicator").style.display = "none";
    }
  };

  socket.onerror = (error) => {
    console.error("WebSocket error:", error);
    stopProcess();
  };

  socket.onclose = () => {
    isProcessing = false;
    document.getElementById("loadingIndicator").style.display = "none";
    document.getElementById("fileLoadingIndicator").style.display = "none";
  };
}

// Process next prompt in the list
function processNextPrompt() {
  if (currentPromptIndex < prompts.length) {
    const currentPrompt = prompts[currentPromptIndex];
    document.getElementById(
      "currentPrompt"
    ).textContent = `Current Prompt: ${currentPrompt}`;
    isProcessing = true;

    // Show loader each time a new prompt starts processing
    document.getElementById("fileLoadingIndicator").style.display =
      "inline-block";

    startWebSocket(currentPrompt); // Send the current prompt
  } else {
    alert("All prompts processed");
    resetProcess();
  }
}

// Display each iteration’s output in the output section
function displayIteration(data) {
  const outputSection = document.getElementById("outputContent");
  const iterationHTML = `
        <div class="iteration">
            <p><strong>Iteration:</strong> ${data.iteration}</p>
            <p><strong>Generated Code:</strong><pre>${
              data.generated_code
            }</pre></p>
            <p><strong>Compiler Output:</strong><pre>${
              data.compiler_output
            }</pre></p>
            <p><strong>Compiled Successfully:</strong> ${
              data.compiled_successfully ? "Yes" : "No"
            }</p>
        </div>
    `;
  outputSection.innerHTML += iterationHTML;
  outputSection.scrollTop = outputSection.scrollHeight;
}

// Stop the current process and keep the current output displayed
function stopProcess() {
  if (socket) socket.close();
  isProcessing = false;
}

// Reset the process, clearing all outputs and re-enabling the input
function resetProcess() {
  stopProcess();
  document.getElementById("userPrompt").value = "";
  document.getElementById("outputContent").innerHTML = "";
  document.getElementById("currentPrompt").textContent = "Current Prompt: None";
  document.getElementById("loadingIndicator").style.display = "none";
  document.getElementById("fileLoadingIndicator").style.display = "none";
  isProcessing = false;
  currentPromptIndex = 0;
}
