package webServer

import (
	"context"
	"fmt"
	"github.com/gorilla/websocket"
	"llama/modules/extraction"
	"llama/program"
	"log"
	"net/http"
	"path/filepath"
)

// WebSocket upgrader
var upgrader = websocket.Upgrader{
	CheckOrigin: func(r *http.Request) bool {
		return true // Allow all origins
	},
}

type UIProgram struct {
	program.Program
}

func (ui *UIProgram) Run() {
	http.HandleFunc("/", serveIndex)
	http.HandleFunc("/ws", ui.handleWebSocket) // WebSocket route
	http.Handle("/static/", http.StripPrefix("/static/", http.FileServer(http.Dir("./static"))))

	fmt.Println("Starting server on http://localhost:8080")
	log.Fatal(http.ListenAndServe(":8080", nil))
}

func serveIndex(w http.ResponseWriter, r *http.Request) {
	http.ServeFile(w, r, filepath.Join("templates", "index.html"))
}

func (ui *UIProgram) handleWebSocket(w http.ResponseWriter, r *http.Request) {
	// Upgrade the HTTP connection to a WebSocket connection
	conn, err := upgrader.Upgrade(w, r, nil)
	if err != nil {
		fmt.Println("Error upgrading to WebSocket:", err)
		return
	}
	defer conn.Close()

	// Create a context with cancellation
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	// Listen for the initial prompt from the client
	var msg map[string]string
	if err := conn.ReadJSON(&msg); err != nil {
		fmt.Println("Error reading JSON:", err)
		return
	}
	userPrompt := msg["prompt"]

	ui.Program.UI = conn // Set the ui to the websocket connection
	ui.Ctx = ctx         // Set the context to the websocket connection

	// Run prog in a separate goroutine and listen for WebSocket close
	go ui.Program.Run(userPrompt, extraction.GoPrompt)

	// Monitor WebSocket for closure to cancel the context
	for {
		_, _, err := conn.ReadMessage()
		if err != nil {
			cancel() // Cancel the context if the connection is closed
			break
		}
	}
}
