package compiler

import (
	"log"
	"os"
	"os/exec"
	"runtime"
	"strings"
)

const TempOutputDir = "tempOutput/"
const TempModuleName = "tempOutput"

type Language string

// Supported languages
const (
	Go   Language = "go"
	Rust Language = "rust"
)

type OS string

// Supported OS
const (
	Windows OS = "windows"
	Linux   OS = "linux"
	MacOS   OS = "darwin" // Darwin is the kernel of macOS
)

// TODO: I want to make an interface for a compilable language, so that we can add more languages in the future
// TODO: The cmd might also be an interface or a struct, so that it can build itself based on the OS and language
// TODO: A cleanup and panic might be needed in setup because if it panics the temp folders should be removed
// TODO: I am not sure that the setup should panic, maybe it should return an error instead so its easier to clean up

type Compiler struct {
	OS            OS
	Language      Language
	languageEnv   ILanguageEnvironment
	SourceCode    string
	Filename      string
	cmdPrefix     string // For example "cmd /c" on Windows
	Dependencies  []string
	tempOutputDir string
}

type ICompiler interface {
	Compile() (string, error)
}

type GoEnvironment struct {
}

// Setup initializes the go environment by creating a go module and running go mod tidy. Panics if it fails.
func (ge *GoEnvironment) SetupEnvironment(cmdPrefix string) {
	// One string
	cmdString := cmdPrefix + " go mod init " + TempModuleName + " && go mod tidy"
	// Split the string into a slice
	cmdSlice := strings.Fields(cmdString) // Fields splits the strings around each instance of one or more consecutive white space characters

	// Make the command
	cmd := exec.Command(cmdSlice[0], cmdSlice[1:]...)
	// Set its target directory
	cmd.Dir = TempOutputDir
	// Execute the command
	err := cmd.Run()
	if err != nil {
		removeTempFolders(TempOutputDir)
		log.Fatalf("Error initializing go module: %v", err)
	}
}

func (ge *GoEnvironment) CheckCompileErrors(filename string, language Language, cmdPrefix string) (string, error) {

	srcCodeFilename := appendSuffix(filename, language)
	compiledFilename := filename

	cmdString := cmdPrefix + " go build -o " + compiledFilename + " " + srcCodeFilename
	cmdSlice := strings.Fields(cmdString) // Fields splits the string on white space of variable length

	cmd := exec.Command(cmdSlice[0], cmdSlice[1:]...)
	cmd.Dir = TempOutputDir
	output, err := cmd.CombinedOutput()
	return string(output), err
}

func InitCompiler(OS OS, language Language, sourceCode string, filename string, dependencies []string) ICompiler {
	compiler := &Compiler{}
	compiler.OS = OS
	compiler.Language = language
	compiler.SourceCode = sourceCode
	compiler.Filename = filename
	compiler.Dependencies = dependencies
	compiler.cmdPrefix = getOsPrefix(OS)
	compiler.languageEnv = getLanguageEnv(language)
	return compiler

}

func getOsPrefix(OS OS) string {
	// Set the cmd prefix based on the OS
	switch OS {
	case Windows:
		return "cmd /c "
	case Linux, MacOS:
		return ""
	default:
		panic("Unsupported OS")
	}
}

func getLanguageEnv(language Language) ILanguageEnvironment {
	switch language {
	case Go:
		return &GoEnvironment{}
	case Rust:
		return &RustEnvironment{}
	default:
		panic("Unsupported language")
	}
}

type ILanguageEnvironment interface {
	SetupEnvironment(cmdPrefix string)
	CheckCompileErrors(filename string, language Language, cmdPrefix string) (string, error)
}

type RustEnvironment struct {
	Compiler
}

// SetupEnvironment initializes the rust environment by creating a cargo project and adding dependencies. Panics if it fails.
func (re *RustEnvironment) SetupEnvironment(cmdPrefix string) {
	// Initialize the rust cargo project--------------------------------------------------------------------------------
	// Command to initialize a cargo project
	cmdString := cmdPrefix + " cargo init --bin " + TempModuleName
	// Split the string into a slice
	cmdSlice := strings.Fields(cmdString)
	// Make the command
	cmd := exec.Command(cmdSlice[0], cmdSlice[1:]...)
	// Set its target directory
	cmd.Dir = TempOutputDir
	// Execute the command
	err := cmd.Run()
	if err != nil {
		log.Fatalf("Error initializing rust project: %v", err)
	}

	// Write the source code to a file----------------------------------------------------------------------------------

	// Create a new file with the source code in the src folder
	// 0644 is the file permission where the user can read and write the file, and the group and others can only read the file.
	err = os.WriteFile(TempOutputDir+TempModuleName+"/src/"+re.Filename, []byte(re.SourceCode), 0644)

	// Update rust dependencies in cargo.toml file using cargo add (cargo-edit)-----------------------------------------
	addCommand := re.cmdPrefix + " cargo add"
	addSlice := strings.Split(addCommand, " ")
	addSlice = append(addSlice, re.Dependencies...)
	err = exec.Command(addSlice[0], addSlice[1:]...).Run()
	if err != nil {
		log.Fatalf("Error adding dependencies: %v", err)
	}
}

func (re *RustEnvironment) CheckCompileErrors(filename string, language Language, cmdPrefix string) (string, error) {

	srcCodeFilename := TempOutputDir + appendSuffix(filename, language)
	output, err := exec.Command(cmdPrefix + " cargo check " + srcCodeFilename).CombinedOutput()
	return string(output), err
}

/*
The compiler pipeline
1. Set up OS and Language
2. Set up the temp folders
3. Write the source code to a file
4. SetupEnvironment the code
5. Return the output and error
*/

func (c *Compiler) Compile() (string, error) {
	// Set up temp folders
	setupTempFolders(TempOutputDir)
	defer removeTempFolders(TempOutputDir)

	srcCodeFilename := TempOutputDir + appendSuffix(c.Filename, c.Language)
	//compiledFilename := TempOutputDir + c.Filename

	// SetupEnvironment either Go or Rust environment, should be an interface method
	c.languageEnv.SetupEnvironment(c.cmdPrefix)

	// Write the source code to a file
	err := os.WriteFile(srcCodeFilename, []byte(c.SourceCode), 0644)
	if err != nil {
		log.Fatalf("Error writing source code to file: %v", err)
	}

	// CheckCompileErrors the code
	return c.languageEnv.CheckCompileErrors(c.Filename, c.Language, c.cmdPrefix)
}

// appendSuffix appends the suffix to the filename if it is not already there depending on the language, panics if the language is not supported
func appendSuffix(filename string, language Language) string {

	suffix := ""
	switch language {
	case Go:
		suffix = ".go"
	case Rust:
		suffix = ".rs"
	default:
		panic("Unsupported language")
	}

	// We check if the filename already has the suffix, if not we append it
	if !strings.HasSuffix(filename, suffix) {
		filename += suffix
	}
	return filename
}

// setupTempFolders creates the temp output directory for compiled files, panics if it fails
func setupTempFolders(tempOutputDir string) {
	// 0777 are the permissions for the directory, everyone can read, write and execute
	err := os.MkdirAll(tempOutputDir, os.ModePerm)
	if err != nil {
		panic("Error creating temp output directory:\n\n" + err.Error())
	}
}

// removeTempFolders removes the temp output directory for compiled files, panics if it fails
func removeTempFolders(tempOutputDir string) {
	err := os.RemoveAll(tempOutputDir)
	if err != nil {
		panic("Error removing temp output directory:\n\n" + err.Error())
	}
}

func CompileStringToGo(code string, filename string, dependencies []string) (string, error) {

	// Get the OS
	os := runtime.GOOS

	// SetupEnvironment
	return InitCompiler(OS(os), Go, code, filename, dependencies).Compile()
}

func CompileStringToRust(code string, filename string, dependencies []string) (string, error) {

	// Get the OS
	os := runtime.GOOS

	// SetupEnvironment
	return InitCompiler(OS(os), Rust, code, filename, dependencies).Compile()
}
