package main

import (
	"bufio"
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"strings"
)

const ollamaEndpoint = "http://localhost:11434/api/generate" // The local endpoint for the Ollama API

// Struct for request to Ollama API
type OllamaRequest struct {
	Prompt  string `json:"prompt"`
	Model   string `json:"model"`
	Context []int  `json:"context,omitempty"` // Context to maintain conversation
}

// Struct for response from Ollama API
type OllamaResponse struct {
	Model              string `json:"model"`
	CreatedAt          string `json:"created_at"`
	Response           string `json:"response"`
	Done               bool   `json:"done"`
	DoneReason         string `json:"done_reason,omitempty"`
	Context            []int  `json:"context,omitempty"` // Updated context
	TotalDuration      int64  `json:"total_duration,omitempty"`
	LoadDuration       int64  `json:"load_duration,omitempty"`
	PromptEvalCount    int    `json:"prompt_eval_count,omitempty"`
	PromptEvalDuration int64  `json:"prompt_eval_duration,omitempty"`
	EvalCount          int    `json:"eval_count,omitempty"`
	EvalDuration       int64  `json:"eval_duration,omitempty"`
}

func main() {
	reader := bufio.NewReader(os.Stdin)
	var conversationContext []int // Variable to store conversation context

	for {

		fmt.Print("Enter your prompt (or type 'exit' to quit): ")
		userPrompt, _ := reader.ReadString('\n')
		userPrompt = strings.TrimSpace(userPrompt)

		if userPrompt == "exit" {
			fmt.Println("Exiting the program.")
			break
		}

		// Generate response using Ollama API, passing the context
		response, updatedContext, err := getOllamaResponse(userPrompt, conversationContext)
		if err != nil {
			fmt.Println("Error generating response:", err)
			continue
		}

		// Update the conversation context with the response
		conversationContext = updatedContext

		fmt.Println("Ollama's response:", response)
	}
}

// Function to make a POST request to Ollama API
func getOllamaResponse(prompt string, context []int) (string, []int, error) {
	// Create request payload with the model specified and context
	requestBody, err := json.Marshal(OllamaRequest{
		Prompt:  prompt,
		Model:   "llama3.1",
		Context: context, // Pass the conversation context
	})
	if err != nil {
		return "", nil, err
	}

	// Send HTTP POST request to Ollama API
	resp, err := http.Post(ollamaEndpoint, "application/json", bytes.NewBuffer(requestBody))
	if err != nil {
		return "", nil, err
	}
	defer resp.Body.Close()

	// Read and accumulate response body in chunks
	var completeResponse string
	var updatedContext []int
	decoder := json.NewDecoder(resp.Body)
	for decoder.More() {
		var chunk OllamaResponse
		if err := decoder.Decode(&chunk); err != nil {
			return "", nil, err
		}
		completeResponse += chunk.Response

		// Capture the updated context from the response
		updatedContext = chunk.Context

		if chunk.Done {
			break
		}
	}

	return completeResponse, updatedContext, nil
}
