package com.application.DB;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.cloud.bigquery.*;

import java.io.File;
import java.io.FileInputStream;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * This class is responsible for handling database related activities
 *
 * @author Eilert Tunheim, Karin Pettersen, Mads Arnesen
 * @version 1.0
 */
public class DB {

    private static Map<String, Number> data;
    private static QueryJobConfiguration queryConfig;
    private static Job queryJob;

    /**
     * Retrieves the credentials file
     * @return
     * @throws Exception
     */
    private static GoogleCredentials getCredentials() throws Exception {
        File credentialsPath = new File("./src/main/resources/com.application/sf-drying-optimization-1e234ad2b0f4.json");

        // Load credentials from JSON key file. If you can't set the GOOGLE_APPLICATION_CREDENTIALS
        // environment variable, you can explicitly load the credentials file to construct the
        // credentials.
        GoogleCredentials credentials;
        try (FileInputStream serviceAccountStream = new FileInputStream(credentialsPath)) {
            credentials = ServiceAccountCredentials.fromStream(serviceAccountStream);
        }
        return credentials;
    }


    /**
     * Creates a builder
     * @return a builder
     * @throws Exception returns potential error
     */
    private static BigQuery getBuilder() throws Exception {
        // Step 1: Initialize BigQuery service
        // Here we set our project ID and get the `BigQuery` service object
        // this is the interface to our BigQuery instance that
        // we use to execute jobs on
        return BigQueryOptions.newBuilder().
                setCredentials(getCredentials()).
                setProjectId("sf-drying-optimization")
                .build().getService();
    }


    /**
     * Creates a job
     * @param queryConfig query configuration information
     * @return a job
     * @throws Exception returns potential error
     */
    private static Job getJob(JobConfiguration queryConfig) throws Exception {
        // Step 3: Run the job on BigQuery
        // create a `Job` instance from the job configuration using the BigQuery service
        // the job starts executing once the `create` method executes

        Job queryJob = getBuilder().create(JobInfo.newBuilder(queryConfig).build());
        queryJob = queryJob.waitFor();
        // the waitFor method blocks until the job completes
        // and returns `null` if the job doesn't exist anymore
        if (queryJob == null) {
            throw new Exception("job no longer exists");
        }
        // once the job is done, check if any error occured
        if (queryJob.getStatus().getError() != null) {
            throw new Exception(queryJob.getStatus().getError().toString());
        }
        return queryJob;
    }


    /**
     * Retrieves information about kwh and the corresponding date
     *
     * @return the results
     * @throws Exception returns potential error
     */
    public static Map<String, Number> getKwh() throws Exception {

        // Initializing a data
        data = new HashMap<>();

        // Preparing a query job
        final String sqlStatement = "SELECT TimeStamp, VariantValue " +
                "FROM sf-drying-optimization.124.int_sd_winccsensordata " +
                "WHERE TimeStamp BETWEEN \"2021-01-25 08:51:03\" " +
                "AND \"2021-03-04 11:10:09\" ORDER BY TimeStamp";

        // Creates a job configuration
        queryConfig = QueryJobConfiguration.newBuilder(sqlStatement).build();

        queryJob = getJob(queryConfig);

        // Step 4: Display results
        // Print out a header line, and iterate through the
        // query results to print each result in a new line
        System.out.println("InTidTork \t UtTidTork");
        TableResult result = queryJob.getQueryResults();
        for (FieldValueList row : result.iterateAll()) {
            // We can use the `get` method along with the column
            // name to get the corresponding row entry

            SimpleDateFormat newFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
            newFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            long timeStamp = row.get("TimeStamp").getTimestampValue()/1000;
            String formatedTimeStamp = newFormat.format(timeStamp);

            int variantValue = row.get("VariantValue").getNumericValue().intValue();

            data.put(formatedTimeStamp, variantValue);

        }
        return data;
    }
}