package com.application.DB;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.cloud.bigquery.*;

import java.io.File;
import java.io.FileInputStream;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.application.DB.Constants.*;

/**
 * This class is responsible for handling database related activities
 *
 * @author Eilert Tunheim, Karin Pettersen, Mads Arnesen
 * @version 1.1
 */
public class DB {

    /**
     * Creates a simple date format to use for converting millis in numbers to a usefull date format
     *
     * @return returns the date format
     */
    private static SimpleDateFormat getDateFormat() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat;
    }

    /**
     * Retrieves the credentials file
     *
     * @return the credentials
     * @throws Exception for potential errors
     */
    private static GoogleCredentials getCredentials() throws Exception {
        File credentialsPath = new File("./src/main/resources/com.application/sf-drying-optimization-1e234ad2b0f4.json");

        // Load credentials from JSON key file. If you can't set the GOOGLE_APPLICATION_CREDENTIALS
        // environment variable, you can explicitly load the credentials file to construct the
        // credentials.
        GoogleCredentials credentials;
        try (FileInputStream serviceAccountStream = new FileInputStream(credentialsPath)) {
            credentials = ServiceAccountCredentials.fromStream(serviceAccountStream);
        }
        return credentials;
    }


    /**
     * Launching the functions
     *
     * @param args default param
     * @throws Exception throws exception in case of error
     */
    public static void main(String[] args) throws Exception {
        getKwh();
        //getName();
        //getZeroPointDate();
    }


    /**
     * Creates a bigquery builder
     *
     * @return a builder
     * @throws Exception returns potential error
     */
    private static BigQuery getBuilder() throws Exception {
        // Step 1: Initialize BigQuery service
        // Here we set our project ID and get the `BigQuery` service object
        // this is the interface to our BigQuery instance that
        // we use to execute jobs on
        return BigQueryOptions.newBuilder().
                setCredentials(getCredentials()).
                setProjectId(PROJECT_ID)
                .build().getService();
    }


    /**
     * Creates a job
     *
     * @param queryConfig query configuration information
     * @return a job
     * @throws Exception returns potential error
     */
    private static Job getJob(JobConfiguration queryConfig) throws Exception {
        // Step 3: Run the job on BigQuery
        // create a `Job` instance from the job configuration using the BigQuery service
        // the job starts executing once the `create` method executes

        Job queryJob = getBuilder().create(JobInfo.newBuilder(queryConfig).build());
        queryJob = queryJob.waitFor();
        // the waitFor method blocks until the job completes
        // and returns `null` if the job doesn't exist anymore
        if (queryJob == null) {
            throw new Exception("job no longer exists");
        }
        // once the job is done, check if any error occured
        if (queryJob.getStatus().getError() != null) {
            throw new Exception(queryJob.getStatus().getError().toString());
        }
        return queryJob;
    }

    /**
     * This function creates a query job that uses the query statement
     * in order to retrieve information from the database
     *
     * @param sqlStatement input for the query statement
     * @return returns the queryjob with the results
     * @throws Exception Throws exception in case of error
     */
    private static TableResult createQueryJob(String sqlStatement) throws Exception {

        // Creates a job configuration
        Job queryJob = getJob(QueryJobConfiguration.newBuilder(sqlStatement).build());

        // Retrieves the results from the queryjob
        return queryJob.getQueryResults();

    }


    /**
     * Retrieves information about kWh and the corresponding date
     *
     * @return the results
     * @throws Exception returns potential error
     */
    public static Map<Integer, Map> getKwh() throws Exception {

        // Initializing the data map to store the results
        Map<Integer, Map> finalResults = new HashMap<>();

        int index = 0;

        for (Map.Entry<String, String> entry : getZeroPointDate().entrySet()) {
            //System.out.printf("Intid: \t%s\t\t\tOuttid: \t%s\n",entry.getKey(),entry.getValue());

            // Initializing the data map to store the results
            Map<String, Number> data = new HashMap<>();

            // Preparing a query statement
            // Query statement 124 Valåsen
            final String sqlStatement = "SELECT `" + TIMESTAMP_NAME + "`, `"+KWH_NAME+"` " +
                    "FROM `" + PROJECT_ID + "." + LOCATION_ID + "." + TABLE_NAME_KWH + "` " +
                    "WHERE " + TIMESTAMP_NAME + " BETWEEN " + '"'+ entry.getKey() + '"' +
                    " AND " + '"' + entry.getValue() + '"' +
                    " AND " + VALUE_ID_NAME + " = " + VALUE_ID + " " +
                    " AND " + KWH_NAME + " <> 0 " +
                    " ORDER BY " + TIMESTAMP_NAME + " ASC";


            System.out.println(sqlStatement);

            // Iterating through the results
            TableResult result = createQueryJob(sqlStatement);

            //System.out.println("Timestamp \t kWh");
            int baseline = 0;
            for (FieldValueList row : result.iterateAll()) {

                // Sets the baseline in order to reset the kWh counter
                if (baseline == 0){
                    baseline = row.get(""+KWH_NAME+"").getNumericValue().intValue();
                }
                //System.out.println("baseline: "+baseline);

                // kWh value
                int variantValue = row.get(""+KWH_NAME+"").getNumericValue().intValue()-baseline; //-baseline

                // Retrieving the wanted data
                long timeStamp = row.get(""+TIMESTAMP_NAME+"").getTimestampValue() / 1000;
                // Riktig format, men i string
                String formatedTimeStamp = getDateFormat().format(timeStamp);

                // Checks for negative values
                if(variantValue > 0) {
                    // Adding the data to a list in order to sort through later
                    data.put(formatedTimeStamp, variantValue);
                }

                //System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\t\t\tBaseline: %s\n",formatedTimeStamp,variantValue,baseline);
                // Checks if the data is empty
                }

            System.out.println("Data size: " + data.size());

            NavigableMap<String, Number> sortedData = new TreeMap<>(data);
            if(!sortedData.isEmpty() && sortedData.size()>50) {
                finalResults.put(index,sortedData);
                index += 1;
            }
        }

        System.out.println("\nFinal results size: "+finalResults.size());

        // Defining a treemap to sort the data incrementally
        NavigableMap<Integer, Map> sortedFinalResults = new TreeMap<>(finalResults);


        for (Map.Entry<Integer, Map> entry : sortedFinalResults.entrySet()) {
            System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\n",entry.getKey(),entry.getValue());
        }

        return sortedFinalResults;
    }


    /**
     * This function retrieves the intidtork and uttidtork dates from the database.
     * This variables are not sorted and thus this function iterates through the data in order to
     * find the last intidtork date and the corresponding uttidtork data.
     * These values will be used to reset the kWh.
     *
     * A possible extention of this function could be to limit the number of dring periodes.
     * This could be done with counting the number of entries in
     *
     * @return Returns a treemap that sorts the Start- and End time for each drying period incrementally
     * @throws Exception Throws exception if an error occurs
     */
    private static Map<String, String> getZeroPointDate() throws Exception{

        // Initializing the dates map to store the results
        Map<String, String> dates = new HashMap<>();

        // Preparing a query statement
        /*
        final String sqlStatement =
                        "SELECT InTidTork, UtTidTork FROM `sf-drying-optimization.124.int_gs_ds_sipalpackages`" +
                        "WHERE Tork Like \"%5%\" AND InTidTork BETWEEN \"2021-01-30 08:51:03\" " +
                                "AND \"2022-03-15 11:10:09\" ORDER BY InTidTork ASC";
         */


        // Defining extra parameters if required
        String extraInputParameter = "";
        if(LOCATION_ID == 124){
            extraInputParameter += "AND CalculatedStart BETWEEN \"1990-01-01 00:00:00\" AND \"" + TODAYS_DATE + "\" ";
        }
        if(LOCATION_ID == 174){
            // KILIN_ID starts at 0 not 1 in the database.
            KILIN_ID -= 1;
        }
        // Input parameters
        if(!TREE_SPECIES.isEmpty()){
            extraInputParameter += "AND " +NAME_PARAMATERS+ " LIKE "+'"'+"%"+ TREE_SPECIES +"%"+'"'+" ";
        }
        if(!WIDTH_DIMENTION.isEmpty()){
            extraInputParameter += "AND " +NAME_PARAMATERS+ " LIKE "+'"'+"%"+ WIDTH_DIMENTION +"%"+'"'+" ";
        }
        if(!HEIGHT_DIMENTION.isEmpty()){
            extraInputParameter += "AND " +NAME_PARAMATERS+ " LIKE "+'"'+"%"+ HEIGHT_DIMENTION +"%"+'"'+" ";
        }
        if(!SAWSET.isEmpty()){
            extraInputParameter += "AND " +NAME_PARAMATERS+ " LIKE "+'"'+"%"+ SAWSET +"%"+'"'+" ";
        }
        if(!MOISTURE_GOAL.isEmpty()){
            extraInputParameter += "AND " +NAME_PARAMATERS+ " LIKE "+'"'+"%"+ MOISTURE_GOAL +"%"+'"'+" ";
        }



        // Sqlstatement for Valåsen(124)
        final String sqlStatement =
                        "SELECT MAX("+ NAME_PARAMATERS +") as DryingSchedule, MAX("+KILIN_NAME+") as Kiln_ID, " + START_DRYING_NAME + ", MAX("+STOP_DRYING_NAME+") as DryingCompleted " +
                        "FROM `" + PROJECT_ID + "." + LOCATION_ID + "." + TABLE_NAME_VALMATICS + "` " +
                        "WHERE " + KILIN_NAME + " = " + KILIN_ID + " " +
                        "AND "+START_DRYING_NAME+" BETWEEN \"1990-01-01 00:00:00\" AND \"" + TODAYS_DATE + "\" " +
                        "AND "+STOP_DRYING_NAME+" BETWEEN \"1990-01-01 00:00:00\" AND \"" + TODAYS_DATE + "\" " +
                                extraInputParameter +
                        "Group by "+ START_DRYING_NAME + " " +
                        "Order by "+ START_DRYING_NAME + " ASC " +
                        "LIMIT " + LIMIT;

        System.out.println(sqlStatement);


        /*
        // Sqlstatement for Årjang(174)
        final String sqlStatement =
                        "SELECT MAX(Name) as DryingSchedule, MAX(KilinId)+1 as KilnName, DryingStarted, Max(DryingCompleted) as DryingCompleted\n" +
                        "FROM `" + PROJECT_ID + "." + LOCATION_ID + "." + TABLE_NAME_VALMATICS + "` " +
                        "WHERE KilinId = 16  \n" +
                        "AND DATE(DryingStarted) BETWEEN \"1990-08-17\" AND \"2022-08-17\"\n" +
                        "AND DATE(DryingCompleted) BETWEEN \"1990-08-17\" AND \"2022-08-17\"\n" +
                        "Group by DryingStarted\n" +
                        "Order by DryingStarted desc";

         */

        // Retrieves the results from the queryjob
        TableResult result = createQueryJob(sqlStatement);

        //System.out.println("InTidTork\t\t\tUtTidTork");
        // Iterating through the results
        for (FieldValueList row : result.iterateAll()) {

            // Defining variables
            String formatedInTidTork = "";
            String formatedUtTidTork = "";

            //System.out.println("Start: "+row.get("DryingStarted").getTimestampValue());
            //System.out.println("Stop: "+row.get("DryingCompleted").getTimestampValue());


            // Retrieving the data
            // DryingStarted:
            if(!row.get("DryingStarted").isNull()){

                // Check if response is given in millis
                try{
                    long doubleValue = row.get("DryingStarted").getTimestampValue();
                    long InTidTorkLong = doubleValue/1000;
                    // Formating the data from long to a string in the correct date format
                    formatedInTidTork = getDateFormat().format(InTidTorkLong);
                    //System.out.println("LONG!!@@@@");
                } catch(NumberFormatException e){
                    //not long value, must be of type string

                    //System.out.println("Must be a string!!@@@@");

                    // stores the value
                    String value = (String) row.get("DryingStarted").getValue();
                    // Splits the string based on 'T'
                    String[] splitValue = value.split("T");
                    // Combines the values into a new format
                    formatedInTidTork = splitValue[0]+" "+splitValue[1];
                }

                // Checks if response is given in a string date format
                //if(row.get("DryingStarted").getValue().equals(Long.parseLong(row.get("DryingStarted").getValue().toString()))) {


            }

            // CalculatedStop:
            // DryingCompleted:
            // Check if response is given in millis
            try{
                long doubleValue = row.get("DryingCompleted").getTimestampValue();
                long utTidTorkLong = doubleValue/1000;
                // Formating the data from long to a string in the correct date format
                formatedUtTidTork = getDateFormat().format(utTidTorkLong);
            } catch(NumberFormatException e) {
                //not long value, must be of type string

                // stores the value
                String value = (String) row.get("DryingCompleted").getValue();
                // Splits the string based on 'T'
                String[] splitValue = value.split("T");
                // Combines the values into a new format
                formatedUtTidTork = splitValue[0] + " " + splitValue[1];
            }

            //System.out.println("Inn formated: "+formatedInTidTork);
            //System.out.println("Ut formated: "+formatedUtTidTork+"\n");

            // Checks if intidtork or outtidtork is empty, if so they are ignored and not added to the list
            if (!formatedInTidTork.isEmpty() && !formatedUtTidTork.isEmpty()){
                // Adds the data to the dates map
                dates.put(formatedInTidTork,formatedUtTidTork);
            }
            //System.out.printf("%s\t\t\t%s\n",formatedInTidTork,formatedUtTidTork);
        }

        // Defining a treemap to sort the data incrementally
        NavigableMap<String, String> sortedFinalResults = new TreeMap<>(dates);

        for (Map.Entry<String, String> entry : sortedFinalResults.entrySet()) {
            System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\n",entry.getKey(),entry.getValue());
        }

        System.out.printf("Size of dates: %s\n\n", dates.size());



        // Returns a treemap that sorts the dates incrementally
        return new TreeMap<>(sortedFinalResults);

/*
        // Defining a treemap to sort through the data
        NavigableMap<String, String> dataSet = new TreeMap<>(dates);

        // Defining a hashmap to store the final sorted data
        Map<String, String> sortedDates = new HashMap<>();

        // Iterating through the data in order to find and set a zeropoint for the dates.
        for (Map.Entry<String, String> entry : dataSet.entrySet()) {

            // Retrieving the entry key and value
            String inTid = entry.getKey();
            String utTid = entry.getValue();

            // Format the date to LocalDateTime datatype
            LocalDateTime date = LocalDateTime.parse(inTid, format);
            // Creates a future date to compare against one day forward in time
            LocalDateTime dateNowPlus = date.plusDays(1);
            String formatedDateNowPlus = format.format(dateNowPlus);

            try {
                // Retrieves the next entry in the list
                String next = dataSet.higherEntry(entry.getKey()).getKey();

                // Checks if the next date is longer than 1 day,
                // if so the current date is the last intidtork date and
                // is the one we will use in order to reset the kWh.
                if(next.compareTo(formatedDateNowPlus) > 0){
                    // Storing the dates that will be used for resetting kWh.
                    sortedDates.put(inTid,utTid);
                }
            } catch (NullPointerException e){
                // Can't find any more iterations, breaks out of the loop.
                break;
            }
        }
        // Defining a treemap to sort through the data
        return new TreeMap<>(sortedDates);

 */
    }
}