package com.application.DB;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.cloud.bigquery.*;

import java.io.File;
import java.io.FileInputStream;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.application.DB.Constants.*;

/**
 * This class is responsible for handling database related activities
 *
 * @author Eilert Tunheim, Karin Pettersen, Mads Arnesen
 * @version 1.1
 */
public class DB {

    /**
     * Creates a simple date format to use for converting millis in numbers to a usefull date format
     *
     * @return returns the date format
     */
    private static SimpleDateFormat getDateFormat() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat;
    }

    /**
     * Retrieves the credentials file
     *
     * @return the credentials
     * @throws Exception for potential errors
     */
    private static GoogleCredentials getCredentials() throws Exception {
        File credentialsPath = new File("./src/main/resources/com.application/sf-drying-optimization-1e234ad2b0f4.json");

        // Load credentials from JSON key file. If you can't set the GOOGLE_APPLICATION_CREDENTIALS
        // environment variable, you can explicitly load the credentials file to construct the
        // credentials.
        GoogleCredentials credentials;
        try (FileInputStream serviceAccountStream = new FileInputStream(credentialsPath)) {
            credentials = ServiceAccountCredentials.fromStream(serviceAccountStream);
        }
        return credentials;
    }


    /**
     * Launching the functions
     *
     * @param args default param
     * @throws Exception throws exception in case of error
     */
    public static void main(String[] args) throws Exception {
        //getKwh();
        //getName();
        //getZeroPointDate();
        System.out.println(setInputParameters());

    }


    /**
     * Creates a bigquery builder
     *
     * @return a builder
     * @throws Exception returns potential error
     */
    private static BigQuery getBuilder() throws Exception {
        // Step 1: Initialize BigQuery service
        // Here we set our project ID and get the `BigQuery` service object
        // this is the interface to our BigQuery instance that
        // we use to execute jobs on
        return BigQueryOptions.newBuilder().
                setCredentials(getCredentials()).
                setProjectId(PROJECT_ID)
                .build().getService();
    }


    /**
     * Creates a job
     *
     * @param queryConfig query configuration information
     * @return a job
     * @throws Exception returns potential error
     */
    private static Job getJob(JobConfiguration queryConfig) throws Exception {
        // Step 3: Run the job on BigQuery
        // create a `Job` instance from the job configuration using the BigQuery service
        // the job starts executing once the `create` method executes

        Job queryJob = getBuilder().create(JobInfo.newBuilder(queryConfig).build());
        queryJob = queryJob.waitFor();
        // the waitFor method blocks until the job completes
        // and returns `null` if the job doesn't exist anymore
        if (queryJob == null) {
            throw new Exception("job no longer exists");
        }
        // once the job is done, check if any error occured
        if (queryJob.getStatus().getError() != null) {
            throw new Exception(queryJob.getStatus().getError().toString());
        }
        return queryJob;
    }

    /**
     * This function creates a query job that uses the query statement
     * in order to retrieve information from the database
     *
     * @param sqlStatement input for the query statement
     * @return returns the queryjob with the results
     * @throws Exception Throws exception in case of error
     */
    private static TableResult createQueryJob(String sqlStatement) throws Exception {

        // Creates a job configuration
        Job queryJob = getJob(QueryJobConfiguration.newBuilder(sqlStatement).build());

        // Retrieves the results from the queryjob
        return queryJob.getQueryResults();

    }


    public static Map<Integer, Map<String, Number>> setInputParameters() throws Exception {

        Map<Integer, Map<String, Number>> allDryingPeriods = new HashMap<>();

        boolean sawsetRemoved = false;
        boolean treespeciesRemoved = false;
        boolean dimensionsRemoved = false;
        boolean moistureRemoved = false;


        HashMap<java.lang.String, java.lang.String> locationTables = null;
        ArrayList<Integer> valmeticsChamberIDS = null;
        ArrayList<Integer> kwhChamberIDS = null;
        int numberOfChambers;

        String extraInputParameter = "";
        int locationID;
        int index = 0;

        String treeSpecies = "";
        String dimensions = "";
        String sawset = "";
        String moistureGoal = "";

        // Valmetics parameters
        String nameParameter;
        String kilinName;
        int kilinID;
        String startDryingTime;
        String stopDryingTime;
        String valmeticsTableName;
        int limit;

        // Kwh parameters
        String kwhTableName;
        String KwhName;
        String timestamp;
        String valueIDName;
        int valueID;


        while(true){

            Map<Integer, Map<String, Number>> intermediateHashMap = new HashMap<>();

            for (HashMap.Entry<Integer, HashMap<String, HashMap<String, String>>> location : Constants.getSawmills().entrySet()) {

                System.out.printf("\nLocation ID: \t%s\t\t\tRest of map: \t%s\n", location.getKey(), location.getValue());
                System.out.println("Number of entries: "+Constants.getSawmills().size());

                //allDryingPeriods = new TreeMap<>();

                locationID = location.getKey();

                //System.out.println(location.getValue());

                if (location.getKey() == 124) {
                    // If location is Valasen, then the database stores furu as fura, swedish.
                    if (TREE_SPECIES.equalsIgnoreCase("Furu")) {
                        TREE_SPECIES = "Fura";
                    }
                    locationTables = getValasenTableNames();
                    valmeticsChamberIDS = getValasenValmeticsChamberIDS();
                    kwhChamberIDS = getValasenKwhChamberIDS();
                }
                if (location.getKey() == 174) {
                    locationTables = getArjangTableNames();
                    valmeticsChamberIDS = getArjangValmeticsChamberIDS();
                    kwhChamberIDS = getArjangKwhChamberIDS();
                }

                // Valmetics
                valmeticsTableName = locationTables.get("Valmetics");
                nameParameter = location.getValue().get(locationTables.get("Valmetics")).get("Name");
                startDryingTime = location.getValue().get(locationTables.get("Valmetics")).get("DryingStarted");
                stopDryingTime = location.getValue().get(locationTables.get("Valmetics")).get("DryingStopped");
                kilinName = location.getValue().get(locationTables.get("Valmetics")).get("KilnName");
                //kilinID = Integer.parseInt(location.getValue().get(locationTables.get("Valmetics")).get("KilinID"));
                limit = Integer.parseInt(location.getValue().get(locationTables.get("Valmetics")).get("Limit"));

                // Kwh
                kwhTableName = locationTables.get("Kwh");
                KwhName = location.getValue().get(locationTables.get("Kwh")).get("KwhName");
                timestamp = location.getValue().get(locationTables.get("Kwh")).get("Timestamp");
                valueIDName = location.getValue().get(locationTables.get("Kwh")).get("ValueIDName");
                //valueID = Integer.parseInt(location.getValue().get(locationTables.get("Kwh")).get("ValueID"));


                System.out.printf("Tree species: \t%s\n", TREE_SPECIES);
                System.out.printf("Width: \t\t\t%s\n", DIMENSIONS);
                System.out.printf("Sawset: \t\t%s\n", SAWSET);
                System.out.printf("Moisture: \t\t%s\n", MOISTURE_GOAL);


                // Defining the extra sql statements for extra parameters
                treeSpecies = "AND LOWER(" + nameParameter + ") LIKE LOWER(" + '"' + "%" + TREE_SPECIES + "%" + '"' + ") ";
                dimensions = "AND LOWER(" + nameParameter + ") LIKE LOWER(" + '"' + "%" + DIMENSIONS + "%" + '"' + ") ";
                sawset = "AND LOWER(" + nameParameter + ") LIKE LOWER(" + '"' + "%" + SAWSET + "%" + '"' + ") ";
                moistureGoal = "AND LOWER(" + nameParameter + ") LIKE LOWER(" + '"' + "%" + MOISTURE_GOAL + "\\\\%" + "%" + '"' + ") ";


                // 1. Checking if user input is empty.
                // 2. Checking if the extraInputParameter variable already contains the extra parameters,
                //    if so not adding a second time.
                // 3. Checking if the while loop have previously removed the parameter, if so, not adding again.
                if (!TREE_SPECIES.isEmpty() && !extraInputParameter.contains(treeSpecies) && !treespeciesRemoved) {
                    extraInputParameter += treeSpecies;
                }
                if (!DIMENSIONS.isEmpty() && !extraInputParameter.contains(dimensions) && !dimensionsRemoved) {
                    extraInputParameter += dimensions;
                }
                if (!SAWSET.isEmpty() && !extraInputParameter.contains(sawset) && !sawsetRemoved) {
                    extraInputParameter += sawset;
                }
                if (!MOISTURE_GOAL.isEmpty() && !extraInputParameter.contains(moistureGoal) && !moistureRemoved) {
                    extraInputParameter += moistureGoal;
                }

                System.out.printf("\nExtra parameters:\n %s\n\n", extraInputParameter);


                // Sets the lowest number of chambers to iterate through
                numberOfChambers = Math.min(valmeticsChamberIDS.size(), kwhChamberIDS.size());

                System.out.printf("\nnumberOfChambers: \t%d\n",numberOfChambers);

                for (int i = 0; i < numberOfChambers; i++) {
                    // Sets the kiln number
                    kilinID = valmeticsChamberIDS.get(i);
                    valueID = kwhChamberIDS.get(i);

                    System.out.printf("KilinID: \t%d\n",kilinID);
                    System.out.printf("valueID: \t%d\n",valueID);

                //System.out.println("Før results");
                Map<Integer, Map<String, Number>> results = new TreeMap<>(getKwh(getZeroPointDate(locationID, extraInputParameter, nameParameter, kilinName, kilinID, startDryingTime, stopDryingTime, valmeticsTableName, limit),
                        locationID, kwhTableName, KwhName, timestamp, valueIDName, valueID, index));

                //System.out.println("\n\nResults size: "+results.size()+"\n\n");
                intermediateHashMap.putAll(results);
                index += results.size();
                //System.out.println("Etter results");
                }
            }


            System.out.printf("\nExtra parameters:\n %s\n\n", extraInputParameter);
            System.out.println("\n\nAll drying periods size: "+intermediateHashMap.size()+"\n\n");
            // Retrieves the dates
            //results = getKwh(getZeroPointDate(locationID, extraInputParameter, nameParameter, kilinName, kilinID, startDryingTime, stopDryingTime, valmeticsTableName, limit));

            // Checks if any dates where found, if not parameters are removed until dates are found
            if (intermediateHashMap.size() < NUMBER_OF_PERIODS) {
                //System.out.println("\nInni if, sletter parameter:\n");
                if (extraInputParameter.contains(sawset) && !sawsetRemoved) {
                    extraInputParameter = extraInputParameter.replace(sawset, "");
                    System.out.println("Sawset is removed");
                    sawsetRemoved = true;
                    //setInputParameters(true,false,false,false);
                } else if (extraInputParameter.contains(treeSpecies) && !treespeciesRemoved) {
                    extraInputParameter = extraInputParameter.replace(treeSpecies, "");
                    System.out.println("Tree species is removed");
                    treespeciesRemoved = true;
                    //setInputParameters(true,true,false,false);
                } else if (extraInputParameter.contains(dimensions) && !dimensionsRemoved) {
                    extraInputParameter = extraInputParameter.replace(dimensions, "");
                    System.out.println("Dimensions is removed");
                    dimensionsRemoved = true;
                    //setInputParameters(true,true,true,false);
                } else if (extraInputParameter.contains(moistureGoal) && !moistureRemoved) {
                    extraInputParameter = extraInputParameter.replace(moistureGoal, "");
                    System.out.println("Moisture goal is removed");
                    moistureRemoved = true;
                    //setInputParameters(true,true,true,true);
                } else break;
            } else {
                // Adding all periods to the hashmap to return
                allDryingPeriods.putAll(intermediateHashMap);
                break;
            }
        }

        System.out.println(allDryingPeriods.size());
        for (Map.Entry<Integer, Map<String, Number>> entry : allDryingPeriods.entrySet()) {
            System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\n", entry.getKey(), entry.getValue());
        }

        return allDryingPeriods;
    }


    /**
     * Retrieves information about kWh and the corresponding date
     *
     * @return the results
     * @throws Exception returns potential error
     */
    public static Map<Integer, Map<String, Number>> getKwh(Map<String, String> dates, int locationID,
                                                           String kwhTableName, String KwhName,
                                                           String timestamp, String valueIDName,
                                                           int valueID, int index) throws Exception {

        // Initializing the data map to store the results
        Map<Integer, Map<String, Number>> finalResults = new HashMap<>();

        for (Map.Entry<String, String> entry : dates.entrySet()) {
            //System.out.printf("Intid: \t%s\t\t\tOuttid: \t%s\n",entry.getKey(),entry.getValue());

            // Initializing the data map to store the results
            Map<String, Number> data = new HashMap<>();

            // Preparing a query statement
            // Query statement 124 Valåsen
            final String sqlStatement = "SELECT `" + timestamp + "`, `" + KwhName + "` " +
                    "FROM `" + PROJECT_ID + "." + locationID + "." + kwhTableName + "` " +
                    "WHERE " + timestamp + " BETWEEN " + '"' + entry.getKey() + '"' +
                    " AND " + '"' + entry.getValue() + '"' +
                    " AND " + valueIDName + " = " + valueID + " " +
                    " AND " + KwhName + " <> 0 " +
                    " ORDER BY " + timestamp + " ASC";


            System.out.println(sqlStatement);

            // Iterating through the results
            TableResult result = createQueryJob(sqlStatement);

            //System.out.println("Timestamp \t kWh");
            int baseline = 0;
            for (FieldValueList row : result.iterateAll()) {

                // Sets the baseline in order to reset the kWh counter
                if (baseline == 0) {
                    baseline = row.get("" + KwhName + "").getNumericValue().intValue();
                }
                //System.out.println("baseline: "+baseline);

                // kWh value
                int variantValue = row.get("" + KwhName + "").getNumericValue().intValue() - baseline; //-baseline

                // Retrieving the wanted data
                long timeStamp = row.get("" + timestamp + "").getTimestampValue() / 1000;
                // Riktig format, men i string
                String formatedTimeStamp = getDateFormat().format(timeStamp);

                // Checks for negative values
                if (variantValue > 0) {
                    // Adding the data to a list in order to sort through later
                    data.put(formatedTimeStamp, variantValue);
                }

                //System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\t\t\tBaseline: %s\n",formatedTimeStamp,variantValue,baseline);
                // Checks if the data is empty
            }

            System.out.println("Data size: " + data.size());

            NavigableMap<String, Number> sortedData = new TreeMap<>(data);
            if (!sortedData.isEmpty() && sortedData.size() > 50) {
                finalResults.put(index, sortedData);
                index += 1;
            }
        }

        System.out.println("\nFinal results size: " + finalResults.size()+"\n\n");

        // Defining a treemap to sort the data incrementally
        NavigableMap<Integer, Map<String, Number>> sortedFinalResults = new TreeMap<>(finalResults);


        for (Map.Entry<Integer, Map<String, Number>> entry : sortedFinalResults.entrySet()) {
            System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\n", entry.getKey(), entry.getValue());
        }

        return sortedFinalResults;
    }


    /**
     * This function retrieves the intidtork and uttidtork dates from the database.
     * This variables are not sorted and thus this function iterates through the data in order to
     * find the last intidtork date and the corresponding uttidtork data.
     * These values will be used to reset the kWh.
     *
     * A possible extention of this function could be to limit the number of dring periodes.
     * This could be done with counting the number of entries in
     *
     * @return Returns a treemap that sorts the Start- and End time for each drying period incrementally
     * @throws Exception Throws exception if an error occurs
     */
    private static Map<String, String> getZeroPointDate(int locationID, String extraUserInput, String nameParameter,
                                                        String kilinName, int kilinID, String startDryingTime,
                                                        String stopDryingTime, String valmeticsTableName, int limit)
                                                        throws Exception {

        // Initializing the dates map to store the results
        Map<String, String> dates = new HashMap<>();


        // Defining extra parameters if required
        String extraInputParameter = "";
        extraInputParameter += extraUserInput;
        if(locationID == 124){
            extraInputParameter += "AND CalculatedStart BETWEEN \"1990-01-01 00:00:00\" AND \"" + TODAYS_DATE + "\" ";
        }
        if(locationID == 174){
            // KILIN_ID starts at 0 not 1 in the database.
            kilinID -= 1;
        }


        // Sqlstatement
        final String sqlStatement =
                        "SELECT MAX("+ nameParameter +") as DryingSchedule, MAX("+kilinName+") as Kiln_ID, " + startDryingTime + ", MAX("+stopDryingTime+") as DryingCompleted " +
                        "FROM `" + PROJECT_ID + "." + locationID + "." + valmeticsTableName + "` " +
                        "WHERE " + kilinName + " = " + kilinID + " " +
                        "AND "+startDryingTime+" BETWEEN \"1990-01-01 00:00:00\" AND \"" + TODAYS_DATE + "\" " +
                        "AND "+stopDryingTime+" BETWEEN \"1990-01-01 00:00:00\" AND \"" + TODAYS_DATE + "\" " +
                                extraInputParameter +
                        "AND LOWER(" +nameParameter+ ") NOT LIKE LOWER("+'"'+"%"+ "test" +"%"+'"'+") " +
                        "Group by "+ startDryingTime + " " +
                        "Order by "+ startDryingTime + " ASC " +
                        "LIMIT " + limit;

        System.out.println(sqlStatement);



        // Retrieves the results from the queryjob
        TableResult result = createQueryJob(sqlStatement);

        //System.out.println("InTidTork\t\t\tUtTidTork");
        // Iterating through the results
        for (FieldValueList row : result.iterateAll()) {

            // Defining variables
            String formatedInTidTork = "";
            String formatedUtTidTork;

            System.out.println(row.get("DryingSchedule").getStringValue());


            //System.out.println("Start: "+row.get("DryingStarted").getTimestampValue());
            //System.out.println("Stop: "+row.get("DryingCompleted").getTimestampValue());


            // Retrieving the data
            // DryingStarted:
            if(!row.get("DryingStarted").isNull()){

                // Check if response is given in millis
                try{
                    long doubleValue = row.get("DryingStarted").getTimestampValue();
                    long InTidTorkLong = doubleValue/1000;
                    // Formating the data from long to a string in the correct date format
                    formatedInTidTork = getDateFormat().format(InTidTorkLong);
                    //System.out.println("LONG!!@@@@");
                } catch(NumberFormatException e){
                    //not long value, must be of type string

                    //System.out.println("Must be a string!!@@@@");

                    if(row.get("DryingStarted").getValue().toString().contains("T")){
                        // stores the value
                        String value = (String) row.get("DryingStarted").getValue();
                        // Splits the string based on 'T'
                        String[] splitValue = value.split("T");
                        // Combines the values into a new format
                        formatedInTidTork = splitValue[0]+" "+splitValue[1];
                    }
                    else formatedInTidTork = row.get("DryingStarted").getValue().toString();
                }

                // Checks if response is given in a string date format
                //if(row.get("DryingStarted").getValue().equals(Long.parseLong(row.get("DryingStarted").getValue().toString()))) {


            }

            // CalculatedStop:
            // DryingCompleted:
            // Check if response is given in millis
            try{
                long doubleValue = row.get("DryingCompleted").getTimestampValue();
                long utTidTorkLong = doubleValue/1000;
                // Formating the data from long to a string in the correct date format
                formatedUtTidTork = getDateFormat().format(utTidTorkLong);
            } catch(NumberFormatException e) {
                //not long value, must be of type string

                if(row.get("DryingCompleted").getValue().toString().contains("T")){
                    // stores the value
                    String value = (String) row.get("DryingCompleted").getValue();
                    // Splits the string based on 'T'
                    String[] splitValue = value.split("T");
                    // Combines the values into a new format
                    formatedUtTidTork = splitValue[0] + " " + splitValue[1];
                }
                else formatedUtTidTork = row.get("DryingCompleted").getValue().toString();

            }

            //System.out.println("Inn formated: "+formatedInTidTork);
            //System.out.println("Ut formated: "+formatedUtTidTork+"\n");

            // Checks if intidtork or outtidtork is empty, if so they are ignored and not added to the list
            if (!formatedInTidTork.isEmpty() && !formatedUtTidTork.isEmpty()){
                // Adds the data to the dates map
                dates.put(formatedInTidTork,formatedUtTidTork);
            }
            //System.out.printf("%s\t\t\t%s\n",formatedInTidTork,formatedUtTidTork);
        }

        // Defining a treemap to sort the data incrementally
        NavigableMap<String, String> sortedFinalResults = new TreeMap<>(dates);

        System.out.println("\n");
/*
        for (Map.Entry<String, String> entry : sortedFinalResults.entrySet()) {
            System.out.printf("Intid: \t%s\t\t\tUttid: \t%s\n",entry.getKey(),entry.getValue());
        }

 */

        System.out.printf("Size of dates: %s\n\n", sortedFinalResults.size());


        // Returns a treemap that sorts the dates incrementally
        return new TreeMap<>(sortedFinalResults);
    }
}
