package com.application.DB;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.cloud.bigquery.*;

import java.io.File;
import java.io.FileInputStream;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * This class is responsible for handling database related activities
 *
 * @author Eilert Tunheim, Karin Pettersen, Mads Arnesen
 * @version 1.0
 */
public class DB {

    private static Job queryJob;

    private static SimpleDateFormat getDateFormat() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat;
    }

    /**
     * Retrieves the credentials file
     *
     * @return the credentials
     * @throws Exception for potential errors
     */
    private static GoogleCredentials getCredentials() throws Exception {
        File credentialsPath = new File("./src/main/resources/com.application/sf-drying-optimization-1e234ad2b0f4.json");

        // Load credentials from JSON key file. If you can't set the GOOGLE_APPLICATION_CREDENTIALS
        // environment variable, you can explicitly load the credentials file to construct the
        // credentials.
        GoogleCredentials credentials;
        try (FileInputStream serviceAccountStream = new FileInputStream(credentialsPath)) {
            credentials = ServiceAccountCredentials.fromStream(serviceAccountStream);
        }
        return credentials;
    }

    public static void main(String[] args) throws Exception {
        getKwh();
        //getName();
        //getZeroPointDate();
    }


    /**
     * Creates a builder
     *
     * @return a builder
     * @throws Exception returns potential error
     */
    private static BigQuery getBuilder() throws Exception {
        // Step 1: Initialize BigQuery service
        // Here we set our project ID and get the `BigQuery` service object
        // this is the interface to our BigQuery instance that
        // we use to execute jobs on
        return BigQueryOptions.newBuilder().
                setCredentials(getCredentials()).
                setProjectId("sf-drying-optimization")
                .build().getService();
    }


    /**
     * Creates a job
     *
     * @param queryConfig query configuration information
     * @return a job
     * @throws Exception returns potential error
     */
    private static Job getJob(JobConfiguration queryConfig) throws Exception {
        // Step 3: Run the job on BigQuery
        // create a `Job` instance from the job configuration using the BigQuery service
        // the job starts executing once the `create` method executes

        Job queryJob = getBuilder().create(JobInfo.newBuilder(queryConfig).build());
        queryJob = queryJob.waitFor();
        // the waitFor method blocks until the job completes
        // and returns `null` if the job doesn't exist anymore
        if (queryJob == null) {
            throw new Exception("job no longer exists");
        }
        // once the job is done, check if any error occured
        if (queryJob.getStatus().getError() != null) {
            throw new Exception(queryJob.getStatus().getError().toString());
        }
        return queryJob;
    }


    /**
     * Retrieves information about kWh and the corresponding date
     *
     * @return the results
     * @throws Exception returns potential error
     */
    public static Map<Integer, Map> getKwh() throws Exception {

        // Initializing the data map to store the results
        Map<Integer, Map> finalResults = new HashMap<>();

        int index = 0;

        /*
        final String sqlStatement = "SELECT TimeStamp, VariantValue " +
                "FROM sf-drying-optimization.124.int_sd_winccsensordata " +
                "WHERE TimeStamp BETWEEN \"2021-01-30 08:51:03\" " +
                "AND \"2021-02-15 11:10:09\" ORDER BY TimeStamp ASC";
         */

        for (Map.Entry<String, String> entry : getZeroPointDate().entrySet()) {
            //System.out.printf("Intid: \t%s\t\t\tOuttid: \t%s\n",entry.getKey(),entry.getValue());

            // Initializing the data map to store the results
            Map<String, Number> data = new HashMap<>();

            // Preparing a query statement
            final String sqlStatement = "SELECT DISTINCT TimeStamp, VariantValue " +
                    "FROM sf-drying-optimization.124.int_sd_winccsensordata " +
                    "WHERE TimeStamp BETWEEN " + '"'+ entry.getKey() + '"' +
                    " AND " + '"' + entry.getValue() + '"' +
                    " ORDER BY TimeStamp ASC";// Preparing a query statement

            //System.out.println(sqlStatement);

            // Creates a job configuration
            queryJob = getJob(QueryJobConfiguration.newBuilder(sqlStatement).build());

            // Iterating through the results
            TableResult result = queryJob.getQueryResults();
            //System.out.println("Timestamp \t kWh");

            int baseline = 0;
            for (FieldValueList row : result.iterateAll()) {

                // Sets the baseline in order to reset the kWh counter
                if (baseline == 0){
                    baseline = row.get("VariantValue").getNumericValue().intValue();
                }
                //System.out.println("baseline: "+baseline);

                // kWh value
                int variantValue = row.get("VariantValue").getNumericValue().intValue()-baseline;

                // Retrieving the wanted data
                long timeStamp = row.get("TimeStamp").getTimestampValue() / 1000;
                // Riktig format, men i string
                String formatedTimeStamp = getDateFormat().format(timeStamp);

                // Checks for negative values and unresonable large values
                if(variantValue > 0 && variantValue < 5000000){
                    // Adding the data to a list in order to sort through later
                    data.put(formatedTimeStamp, variantValue);
                }
                //System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\t\t\tBaseline: %s\n",formatedTimeStamp,variantValue,baseline);
                // Checks if the data is empty
                }
            NavigableMap<String, Number> sortedData = new TreeMap<>(data);
            if(!sortedData.isEmpty()) {
                finalResults.put(index,sortedData);
                index += 1;
            }
        }
        // Defining a treemap to sort through the data
        NavigableMap<Integer, Map> sortedFinalResults = new TreeMap<>(finalResults);


        for (Map.Entry<Integer, Map> entry : sortedFinalResults.entrySet()) {
            System.out.printf("Timestamp: \t%s\t\t\tkWh: \t%s\n",entry.getKey(),entry.getValue());
        }

        return sortedFinalResults;
    }


    /**
     * Retrieves information about kWh and the corresponding date
     *
     * @throws Exception returns potential error
     */
    public static void getName() throws Exception {

        // Initializing the data map to store the results
        Map<Object,ArrayList<Object>> data = new HashMap<>();


        // Preparing a query statement
        final String sqlStatement =
                        "SELECT Name, CalculatedStart, CalculatedStop FROM `sf-drying-optimization.124.int_dk_valmaticsdryingbatches`" +
                        "WHERE Name Like \"%Gran%\" AND Name Like \"%3ex%\" AND NAME Like \"%47x150%\" AND DATE(CalculatedStart) " +
                        "BETWEEN \"2018-08-17\" AND \"2022-08-30\" ORDER BY DATE(CalculatedStart)";

        // Creates a job configuration
        queryJob = getJob(QueryJobConfiguration.newBuilder(sqlStatement).build());

        // Iterating through the results
        TableResult result = queryJob.getQueryResults();
        System.out.println("Name\tCalculatedStarted\tCalculatedStop");
        for (FieldValueList row : result.iterateAll()) {

            // Retrieving the wanted data
            String name = row.get("Name").getStringValue();
            // The dates are returned as a 16-digit number that needs to be formatted
            long calculatedStart = row.get("CalculatedStart").getTimestampValue()/1000;
            long calculatedStop = row.get("CalculatedStop").getTimestampValue()/1000;
            // Formatting the dates
            String formattedCalculatedStart = getDateFormat().format(calculatedStart);
            String formattedCalculatedStop = getDateFormat().format(calculatedStop);

            java.sql.Timestamp timestamp = new Timestamp(calculatedStart);
            //System.out.println(timestamp);

            System.out.printf("%s\t\t\t%s\t\t\t%s\n",name, formattedCalculatedStart, formattedCalculatedStop);

        }
        //return data;
    }

    /**
     * This function retrieves the intidtork and uttidtork dates from the database.
     * This variables are not sorted and thus this function iterates through the data in order to
     * find the last intidtork date and the corresponding uttidtork data.
     * These values will be used to reset the kWh.
     *
     * A possible extention of this function could be to limit the number of dring periodes.
     * This could be done with counting the number of entries in
     *
     * @return A hashmap containing the correct intidtork and outtidtork dates
     * @throws Exception Throws exception if an error occurs
     */
    private static Map<String, String> getZeroPointDate() throws Exception{

        // Initializing a date format in the data type DateTimeFormatter
        // Required for iterating through the dates.
        DateTimeFormatter format = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

        // Initializing the dates map to store the results
        Map<String, String> dates = new HashMap<>();

        // Preparing a query statement
        final String sqlStatement =
                        "SELECT InTidTork, UtTidTork FROM `sf-drying-optimization.124.int_gs_ds_sipalpackages`" +
                        "WHERE Tork Like \"%5%\" AND InTidTork BETWEEN \"2021-01-30 08:51:03\" " +
                                "AND \"2022-03-15 11:10:09\" ORDER BY InTidTork ASC";

        // Creates a job configuration
        queryJob = getJob(QueryJobConfiguration.newBuilder(sqlStatement).build());

        // Retrieves the results from the queryjob
        TableResult result = queryJob.getQueryResults();

        //System.out.println("InTidTork\t\t\tUtTidTork");

        // Iterating through the results
        for (FieldValueList row : result.iterateAll()) {

            // Defining variables
            String formatedInTidTork;
            String formatedUtTidTork;

            // Retrieving the data
            // InTidTork:
            if(!row.get("InTidTork").isNull()){
                long InTidTorkLong = row.get("InTidTork").getTimestampValue()/1000;
                // Formating the data from long to a string in the correct date format
                formatedInTidTork = getDateFormat().format(InTidTorkLong);
            } else {
                formatedInTidTork = "";
            }

            // UtTidTork:
            if(!row.get("UtTidTork").isNull()){
                long utTidTorkLong = row.get("UtTidTork").getTimestampValue()/1000;
                // Formating the data from long to a string in the correct date format
                formatedUtTidTork = getDateFormat().format(utTidTorkLong);
            } else {
                formatedUtTidTork = "";
            }

            // Checks if intidtork or outtidtork is empty, if so they are ignored and not added to the list
            if (!formatedInTidTork.equals("") && !formatedUtTidTork.equals("")){
                // Adds the data to the dates map
                dates.put(formatedInTidTork,formatedUtTidTork);
            }
            //System.out.printf("%s\t\t\t%s\n",formatedInTidTork,formatedUtTidTork);
        }

        // Defining a treemap to sort through the data
        NavigableMap<String, String> dataSet = new TreeMap<>(dates);

        // Defining a hashmap to store the final sorted data
        Map<String, String> sortedDates = new HashMap<>();

        // Iterating through the data in order to find and set a zeropoint for the dates.
        for (Map.Entry<String, String> entry : dataSet.entrySet()) {

            // Retrieving the entry key and value
            String inTid = entry.getKey();
            String utTid = entry.getValue();

            // Format the date to LocalDateTime datatype
            LocalDateTime date = LocalDateTime.parse(inTid, format);
            // Creates a future date to compare against one day forward in time
            LocalDateTime dateNowPlus = date.plusDays(1);
            String formatedDateNowPlus = format.format(dateNowPlus);

            try {
                // Retrieves the next entry in the list
                String next = dataSet.higherEntry(entry.getKey()).getKey();

                // Checks if the next date is longer than 1 day,
                // if so the current date is the last intidtork date and
                // is the one we will use in order to reset the kWh.
                if(next.compareTo(formatedDateNowPlus) > 0){
                    // Storing the dates that will be used for resetting kWh.
                    sortedDates.put(inTid,utTid);
                }
            } catch (NullPointerException e){
                // Can't find any more iterations, breaks out of the loop.
                break;
            }
        }
        // Defining a treemap to sort through the data
        return new TreeMap<>(sortedDates);
    }
}