library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

library work;
use work.fifo;

entity Rx is
    generic(
        F_CLK_115k: integer:= 54;
        F_CLK_576: integer:= 109;
        F_CLK_384: integer:= 162;
        F_CLK_192: integer:= 325;
        F_CLK_096: integer:= 650
    );
    Port (
        clk: in  std_logic;  -- Systemklokke
        rst_n: in  std_logic;  -- Reset signal
        RxD: in std_logic;
		Data_bus : inout std_logic_vector(7 downto 0);
        Rd_sig: in std_logic;
        Wr_sig: in std_logic;
        adresse: in std_logic_vector(2 downto 0)
    );
end Rx;

architecture Behavioral of Rx is
    --------------------------------------------------------------------------------------------------------------------------------
    --Datatyper
    --------------------------------------------------------------------------------------------------------------------------------
    type paritets_sjekk is (ingen, partall, oddetall);
    type baud_speed is (B115k,B576,B384,B192,B096);
    --------------------------------------------------------------------------------------------------------------------------------
    --Sigwork
    --------------------------------------------------------------------------------------------------------------------------------
    signal datalest: std_logic_vector(10 downto 0):="11111111111";
    signal datasamplet: std_logic_vector(7 downto 0):="00000000";
    signal teller_clk: natural range 0 to F_CLK_096;
    signal teller_sample: natural range 0 to 7;
    signal parity_check: paritets_sjekk:= ingen;
    signal baud_rate: baud_speed := B115k;
    signal fifo_inn: std_logic_vector(7 downto 0);
    signal paritetsfeil: std_logic := '0';
    signal datalost: std_logic := '0';
    signal fifofull: std_logic := '0';
    signal fifo_ut: std_logic_vector(7 downto 0);
    signal fifotom: std_logic := '1';
    signal fifo_rst: std_logic := '0';
    signal fifo_rd_req: std_logic:= '0';
    signal fifo_wr_req: std_logic:= '0';

    --------------------------------------------------------------------------------------------------------------------------------
    --Funksjoner
    --------------------------------------------------------------------------------------------------------------------------------

    --------------------------------------------------------------------------------------------------------------------------------
    --bitsjekk ser på det samplede RxD signalet, og ser om den skal returnere 0 eller 1 basert på antall 0'ere og 1'ere
    --------------------------------------------------------------------------------------------------------------------------------
    function bitsjekk(datasamplet: std_logic_vector) return std_logic is 
        variable datasample_read: std_logic_vector(5 downto 0);
        variable sum: natural range 0 to 8 := 0;
    begin
        datasample_read(5 downto 0) := datasamplet(6 downto 1);
        
        for i in datasample_read'range loop
            if(datasample_read(i) = '1') then
                sum := sum + 1;
            else
                sum := sum;
            end if;
        end loop;
        if sum <= 3 then 
            return '0';
        else
            return '1';
        end if;
    end function;

    ------------------------------------------------------------------------------------------------------------------------------
    --parity_checker ser på hele den leste datastrengen, og avgjør om man skal forkaste dataen eller ikke
    ------------------------------------------------------------------------------------------------------------------------------
    function parity_checker(parity_check:paritets_sjekk; datalest:std_logic_vector(10 downto 0)) return boolean is
        variable datalest_parity: std_logic_vector(9 downto 1) := datalest(9 downto 1);
        variable sum_0 : natural range 0 to 11 := 0;
        variable sum_1 : natural range 0 to 11 := 0;
    begin
        for i in datalest_parity'range loop
                    if(datalest_parity(i) = '0') then
                        sum_1 := sum_1 + 1;
                    else
                        sum_0 := sum_0 + 1;
                    end if;
        end loop;

        case parity_check is
            when partall =>                                                                                         -- når antall 1'ere og nuller er like, returner true, hvis ikke, returner false
                if sum_0 = sum_1 then
                    return true;
                else 
                    return false;
                end if;                                                                                                                                                                                          
            when oddetall =>                                                                                         -- når antall 1'ere og nullerikke er like, returner true, hvis ikke, returner false
            if sum_0 = sum_1 then
                return false;
            else 
                return true;
            end if;
            when others =>
                return true;
        end case;
    end function;

    -----------------------------------------------------------------------------------------------------------------------------
    --baud_rate_teller returnerer en tellerverdi, basert på den satte baudraten ----------- type baud_speed(B115k,B576,B384,B192,B096);
    -----------------------------------------------------------------------------------------------------------------------------
    function baud_rate_teller(baud_rate:baud_speed) return integer is
    begin
        case baud_rate is
            when B115k => 
                return F_CLK_115k;
            when B576 => 
                return F_CLK_576;
            when B384 =>
                return F_CLK_384;
            when B192 =>
                return F_CLK_192;
            when B096 =>
                return F_CLK_096;
            when others =>
                return F_CLK_115k;
        end case;
    end function;

    -------------------------------------------------------------------------------------------------------------------------------
    --parity_config tar bit 4 og 3 fra databussen, og returnerer hvilken partitetssjekk som skal benyttes
    -------------------------------------------------------------------------------------------------------------------------------
    function parity_config(Data_bus:std_logic_vector) return paritets_sjekk is
    begin
        case Data_bus(4 downto 3) is
            when "00" =>
                return ingen;
            when "10" =>
                return partall;
            when "11" =>
                return oddetall;
            when others =>
                return ingen;
        end case;
    end function;

    impure function baud_config(Databus:std_logic_vector) return baud_speed is
    begin
        case Data_bus(2 downto 0) is
            when "000" => 
                return B115k;
            when "001" =>
                return B576;
            when "010" =>
                return B192;
            when "100" =>
                return B096;
            when others =>
                return B115k;
        end case;
    end function;
begin

    --------------------------------------------------------------------------------------------------------------------------------
    --component fifo
    --------------------------------------------------------------------------------------------------------------------------------
    fifo: entity work.fifo 
        Port map(
            clock => clk,
            data => fifo_inn,
            empty => fifotom,
            full => fifofull,
            q => fifo_ut,
            sclr => fifo_rst,
            rdreq => fifo_rd_req,
            wrreq => fifo_wr_req
        );

    -------------------------------------------------------------------------------------------------------------------------------
    --Prosesser
    -------------------------------------------------------------------------------------------------------------------------------

    -------------------------------------------------------------------------------------------------------------------------------
    --sample prosessen fungerer som et shif-register. Den sampler 8 ganger i forhold til den gitte baudraten, og
    --flytter en bit med den nye biten som blir samplet mtp bitsjekk funksjonen.
    -------------------------------------------------------------------------------------------------------------------------------
    sample: process(clk,rst_n) is
        begin
            if rst_n = '0' then
            
            elsif rising_edge(clk) then
                if teller_clk /= baud_rate_teller(baud_rate) - 1 then                           --Sett inn funksjon for bestemmelse av teller
                    teller_clk <= teller_clk + 1;
                else                                                                            --Sett inn logikk for sampling 
                    if teller_sample /= 7 then --hello
                        datasamplet <= datasamplet(7 downto 1) & RxD;                           --shift register for å lese data
                        teller_sample <= teller_sample + 1;
                    else
                        datalest <= datalest(10 downto 1) & bitsjekk(datasamplet);
                        teller_sample <= 0;
                    end if;
                end if;
            end if;
    end process;
    -------------------------------------------------------------------------------------------------------------------------------
    --databehandling avgjør om man skal sende dataen inn til fifoen eller forkaste den. Dette gjøres ved å først sjekke om det er
    --en startbit, før den bruker parity_checker funksjonen mtp hvilken paritetssjekk man har. Hvis parity-checker returnerer true
    --sendes dataen inn i fifoen, hvis ikke så forkastes den.
    -------------------------------------------------------------------------------------------------------------------------------
databehandling: process(clk, rst_n)
begin
    if rst_n = '0' then
        datalest(10 downto 0) <= (others => '0');
        fifo_wr_req <= '0';  -- Nullstille fifo_wr_req under reset
        paritetsfeil <= '0'; -- Nullstille paritetsfeil under reset
    elsif rising_edge(clk) then
        if datalest(10) = '0' then
            if parity_checker(parity_check, datalest) = true then
                fifo_wr_req <= '1';  -- Aktiver fifo_wr_req når paritetskontroll er gyldig
                case parity_check is
                    when ingen =>
                        fifo_inn <= datalest(9 downto 2);
                        datalest(10 downto 1) <= (others => '1');
                    when others =>
                        fifo_inn <= datalest(9 downto 2);
                        datalest(10 downto 0) <= (others => '1');
                end case;
                paritetsfeil <= '0';  -- Nullstill paritetsfeil
            else
                datalest <= (others => '1');  -- Feilhåndtering
                paritetsfeil <= '1';
                fifo_wr_req <= '0';  -- Deaktiver fifo_wr_req ved feil
            end if;
        else
            fifo_wr_req <= '0';  -- Deaktiver fifo_wr_req hvis datalest(10) ikke er '0'
        end if;
    end if;
end process;
    
    kommunikasjon: process(clk, rst_n) is
        begin
            if rst_n = '0' then
            fifo_rd_req <= '0';
            elsif rising_edge(clk) then
                case adresse is
                    when "100" =>
                        parity_check <= parity_config(Data_bus);
                        baud_rate <= baud_config(Data_bus);
                    when "101" =>
                        if Rd_sig = '1' then
                            Data_bus <= fifo_ut;
                            fifo_rd_req <= '1';
                        else
                            null;
                        end if;
                    when "110" =>
                        if Wr_sig = '1' then
                            Data_bus <= "0000" & paritetsfeil & datalost & fifofull & fifotom;
                        else
                            null;    
                        end if;
                    when others =>
                    Data_bus <= (others => 'Z'); 
                end case;
            end if;
    end process;
end Behavioral; 
